<?php
/**
 * WPMR_Definitions_Trait
 *
 * Generated during Phase 2 restructuring
 * This trait contains methods extracted from the original monolithic wpmr.php
 *
 * @package WP_Malware_Removal
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;

}

trait WPMR_Definitions {

	function maybe_load_default_definitions() {
		$definitions = $this->get_setting( 'signatures' );
		if ( ! $definitions ) {
			$definitions = file_get_contents( trailingslashit( __DIR__ ) . 'wpmr.json' );
			$definitions = json_decode( $definitions, true );
			$this->update_setting( 'signatures', $definitions );
			$this->update_setting( 'sig_time', 0 );
			return $definitions;
		}
	}

	function get_definitions() {
		$definitions = $this->get_setting( 'signatures' );
		if ( ! $definitions ) {
			$definitions = $this->maybe_load_default_definitions();
		}
		unset( $definitions['v'] );
		$severe     = array();
		$high       = array();
		$suspicious = array();
		foreach ( $definitions['definitions']['files'] as $definition => $signature ) {
			if ( $signature['severity'] == 'severe' ) {
				$severe[ $definition ] = $definitions['definitions']['files'][ $definition ];
			}
			if ( $signature['severity'] == 'high' ) {
				$high[ $definition ] = $definitions['definitions']['files'][ $definition ];
			}
			if ( $signature['severity'] == 'suspicious' ) {
				$suspicious[ $definition ] = $definitions['definitions']['files'][ $definition ];
			}
		}
		$files      = array_merge( $severe, $high, $suspicious ); // always return definitions in this sequence else suspicious matches are returned first without scanning for severe infections.
		$severe     = array();
		$high       = array();
		$suspicious = array();
		foreach ( $definitions['definitions']['db'] as $definition => $signature ) {
			if ( $signature['severity'] == 'severe' ) {
				$severe[ $definition ] = $definitions['definitions']['db'][ $definition ];
			}
			if ( $signature['severity'] == 'high' ) {
				$high[ $definition ] = $definitions['definitions']['db'][ $definition ];
			}
			if ( $signature['severity'] == 'suspicious' ) {
				$suspicious[ $definition ] = $definitions['definitions']['db'][ $definition ];
			}
		}
		$db                                  = array_merge( $severe, $high, $suspicious );
		$definitions['definitions']['files'] = $files;
		$definitions['definitions']['db']    = $db;
		return $definitions;
	}

	function get_definition_count() {
		$defs  = $this->get_definitions();
		$count = 0;
		while ( count( $defs['definitions'] ) ) {
			$count += count( array_shift( $defs['definitions'] ) );
		}
		return $count;
	}

	function get_definition_version() {
		$sigs = $this->get_setting( 'signatures' );
		if ( ! empty( $sigs ) && ! empty( $sigs['v'] ) ) {
			return $sigs['v'];
		}
	}

	function get_last_updated_ago() {
		$updated = $this->get_setting( 'sig_time' );
		if ( ! $updated ) {
			return 'Never';
		} else {
			return human_time_diff( gmdate( 'U', $updated ), gmdate( 'U' ) ) . ' ago';
		}
	}

	function get_definitions_update_url() {
		$url           = WPMR_SERVER;
		$args          = array(
			'cachebust'   => time(),
			'wpmr_action' => 'update-definitions',
		);
		$compatibility = $this->plugin_data;
		$state         = $this->get_setting( 'user' );
		$lic           = $this->get_setting( 'license_key' );
		if ( $state ) {
			$state = array_merge( $state, $compatibility );
		} else {
			$state = $compatibility;
		}
		if ( $lic ) {
			$state['lic'] = $lic;
		}
		$args['state'] = $this->encode( $state );
		return trailingslashit( $url ) . '?' . urldecode( http_build_query( $args ) );
	}

	function check_definitions( $async = false ) {
		$blocking = empty( $async );
		if ( $blocking ) {
			$timeout = $this->timeout;
		} else {
			$timeout = 0.01;
		}
		$response    = wp_safe_remote_request(
			$this->get_definitions_check_url(),
			array(
				'timeout'     => $timeout,
				'httpversion' => '1.1',
				'blocking'    => $blocking,
			)
		);
		$headers     = wp_remote_retrieve_headers( $response );
		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 != $status_code ) {
			return;
		}
		if ( is_wp_error( $response ) ) {
			return;
		}
		$body    = wp_remote_retrieve_body( $response );
		$version = json_decode( $body, true );
		if ( is_null( $version ) ) {
			return;
		}
		if ( $version['success'] != true ) {
			return;
		}
		if ( ! empty( $version['success'] ) && $version['success'] == true ) {
			$version = $version['data'];
			$time    = gmdate( 'U' );
			$this->update_setting( 'update-version', $version );
			return true;
		}
	}

	function get_definitions_check_url() {
		$url           = WPMR_SERVER;
		$args          = array(
			'cachebust'   => time(),
			'wpmr_action' => 'check-definitions',
		);
		$compatibility = $this->plugin_data;
		$state         = $this->get_setting( 'user' );
		if ( $state ) {
			$state = array_merge( $state, $compatibility );
		} else {
			$state = $compatibility;
		}
		$state            = array_merge( $state, array( 'defver' => $this->get_definition_version() ) );
		$args['state']    = $this->encode( $state );
		$update_check_url = trailingslashit( $url ) . '?' . urldecode( http_build_query( $args ) );
		return $update_check_url;
	}

	function update_definitions( $force = false ) {
		check_ajax_referer( 'wpmr_update_sigs', 'wpmr_update_nonce' );
		if ( ! current_user_can( $this->cap ) ) {
			return;
		}
		$this->raise_limits_conditionally();
		$response    = wp_safe_remote_request(
			$this->get_definitions_update_url(),
			array(
				'timeout'     => $this->timeout,
				'httpversion' => '1.1',
			)
		);
		$headers     = wp_remote_retrieve_headers( $response );
		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 != $status_code ) {
			return wp_send_json_error( 'Error ' . $status_code . ' fetching Update.' );
		}
		if ( is_wp_error( $response ) ) {
			return wp_send_json_error( $response->get_error_message() );
		}
		$body        = wp_remote_retrieve_body( $response );
		$definitions = json_decode( $body, true );
		if ( is_null( $definitions ) ) {
			return wp_send_json_error( 'Unparsable definition-update.' );
		}
		if ( $definitions['success'] != true ) {
			return wp_send_json_error( sanitize_text_field( $definitions['data'] ) );
		}
		if ( ! empty( $definitions['success'] ) && $definitions['success'] == true ) {
			$definitions = $definitions['data'];
			$this->update_setting( 'signatures', $definitions );
			$time = gmdate( 'U' );
			$this->update_setting( 'sig_time', $time );
			return wp_send_json_success(
				array(
					'count'    => $this->get_definition_count(),
					'version'  => $this->get_definition_version(),
					'sig_time' => $this->get_last_updated_ago(),
				)
			);
		}
		return wp_send_json_error( 'Unknown error.' );
	}

	function update_definitions_cli( $echo = false ) {
		$this->raise_limits_conditionally();
		$response    = wp_safe_remote_request(
			$this->get_definitions_update_url(),
			array(
				'timeout' => $this->timeout,
			)
		);
		$headers     = wp_remote_retrieve_headers( $response );
		$status_code = wp_remote_retrieve_response_code( $response );

		if ( 200 != $status_code ) {
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::error( 'Error ' . $status_code . ' fetching Update.' );
				} else {
					echo 'Error ' . esc_html( $status_code ) . ' fetching Update.';
				}
			} else {
				return false;
			}
		}
		if ( is_wp_error( $response ) ) {
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::error( $response->get_error_message() );
				} else {
					echo 'Error: ' . esc_html( $response->get_error_message() );
				}
			} else {
				return false;
			}
		}
		$body        = wp_remote_retrieve_body( $response );
		$definitions = json_decode( $body, true );
		if ( is_null( $definitions ) ) {
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::error( 'Unparsable definition-update.' );
				} else {
					echo 'Unparsable definition-update.';
				}
			} else {
				return false;
			}
		}
		if ( $definitions['success'] != true ) {
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::error( sanitize_text_field( $definitions['data'] ) );
				} else {
					echo esc_html( sanitize_text_field( $definitions['data'] ) );
				}
			} else {
				return false;
			}
		}
		if ( ! empty( $definitions['success'] ) && $definitions['success'] == true ) {
			$definitions = $definitions['data'];
			$this->update_setting( 'signatures', $definitions );
			$time = gmdate( 'U' );
			$this->update_setting( 'sig_time', $time );
			if ( $echo ) {
				if ( $this->wpmr_iscli() ) {
					WP_CLI::success( 'Updated Malcure definitions to version: ' . WP_CLI::colorize( '%Y' . $definitions['v'] . '. %nCount: %Y' . $this->get_definition_count() . '%n' ) . ' definitions.' );
				} else {
					echo 'Updated Malcure definitions to version <strong>' . esc_html( $definitions['v'] ) . '</strong>. Count: <strong>' . esc_html( $this->get_definition_count() ) . '</strong> definitions.';
				}
			} else {
				return true;
			}
		}
	}

	function definition_updates_available() {
		$current = $this->get_definition_version();
		$new     = $this->get_setting( 'update-version' );
		// $this->flog( 'Definition versions: current=' . $current . '= new=' . $new . '=' );
		if ( ! empty( $current ) && ! empty( $new ) && $current != $new ) {
			return array(
				'new'     => $new,
				'current' => $current,
			);
		}
	}
}
